%psDecomposition    Get pure-state decomposition of a density matrix.
%   [PSI P] = psDecomposition(U, CHI, LAMBDA) returns the pure-state decomposition of
%   an n x n density matrix RHO with rank r. The decomposition is parameterized 
%   by the m x r complex Stiefel matrix U, where m >= r. The columns of the 
%   n x r matrix CHI must contain the normalized eigenvectors of RHO, while 
%   the vector LAMBDA holds the corresponding r positive eigenvalues.
%   The columns of PSI contain the state vectors of the decomposition, and
%   the vector P contains the corresponding probabilities.
%   States with probability smaller than eps are set to [1; 0; ...; 0] and
%   the corresponding probability is set to zero.
%
%   See also: densityEig

% Copyright (C) 2011 Beat Röthlisberger
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/

function [PSI, P] = psDecomposition(U, CHI, LAMBDA)

PSI = CHI*diag(sqrt(LAMBDA))*transpose(U);
P = diag(PSI'*PSI);

% Set entries in P < eps to zero
P(P < eps) = 0;

% normalize the vectors PSI.
PSI = PSI*diag(sqrt(1./P));

% set vectors corresponding to zero probability to some harmless standard
% state
PSI(:, P == 0) = kron(ones(1, sum(P == 0)), [1; zeros(size(PSI, 1) - 1, 1)]);
