%decomposeUnitary    Get parameterization of a Stiefel matrix.
%   X = decomposeUnitary(U) calculates a set of angles parameterizing the
%   Stiefel matrix U. It can be considered as the inverse of buildUnitary.
%
%   See also: buildUnitary

% Copyright (C) 2011 Beat Röthlisberger
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>

function X = decomposeUnitary(U)

    [m, n] = size(U);

    cnt = 1;

    for i = 1:n
        for k = m:-1:(i+1)
        
            u1 = U(k-1, i);
            u2 = U(k  , i);
        
            if abs(u1) < eps
            
                t = pi/2;
                p = 0;
            
            else
            
                t = atan(abs(u2)/abs(u1));
                p = (angle(u2) - angle(u1))/2; 
            end
        
            U = givens(m, k-1, k, t, p)*U;
       
            theta(cnt) = t;
            phi(cnt)   = p;
        
            cnt        = cnt + 1;
        end
    end

    test_norm = norm(U'*U - eye(n));
    
    if test_norm > 1e-11
        error(['decomposition failed: norm = ', num2str(test_norm)]);
    end

    if (n == 1)
    
        chi = angle(U(1));
    else
        chi = angle(diag(U)).';
    end

    % reverse theta and phi, so that they are in the proper order for
    % rebuilding the matrix
    l = length(theta);
    theta = theta(l:-1:1);
    phi   = phi(l:-1:1);

    X = [theta, phi, chi]';
end




% Givens Rotation
%
% Note that there is only one phase angle, phi
% The result is a sparse matrix.
function G = givens(n, i, k, theta, phi)

    G = speye(n);

    G(i, i) =  cos(theta)*exp( 1i*phi);
    G(i, k) =  sin(theta)*exp(-1i*phi);
    G(k, i) = -sin(theta)*exp( 1i*phi);
    G(k, k) =  cos(theta)*exp(-1i*phi);
end


