%bfgs_min   BFGS quasi-Newton minimization.
%   [f, X, info] = bfgs_min(FUN, GRAD, X0) performs a BFGS quasi-Newton
%   minimization of the function FUN over Euclidean space. GRAD must
%   return the derivatives of FUN with respect to X, where X is a real
%   vector and is FUN's and GRAD's only argument. The search is started at X0.
%
%   [f, X, info] = bfgs_min(FUN, GRAD, X0, params) is used to specify
%   termination parameters different from the default values. These
%   parameters in the struct params are (with default values in parentheses):
%
%   params.MaxIter (1000) Maximum number of iterations
%   params.TolFun (1e-12) Minimum difference between two consecutive
%                         function values
%   params.TolG (1e-10)   Minimum difference between two consecutive
%                         gradient norms
%   params.TolX (1e-10)   Minimum difference between two consecutive search 
%                         point norms
%
%   The return values f and X store the minimum function value and the
%   corresponding point in the search space, respectively. The struct 
%   info contains:
%
%   info.fvals   The function values encountered during the minimization
%   info.xvals   A cell array containing the corresponding search points
%   info.status  Termination status, where
%
%                -2 = Too many iterations
%                -1 = Line search could not make further progress
%                 0 = Terminated on gradient tolerance
%                 1 = Terminated on function tolerance
%                 2 = Terminated on search point tolerance

% Copyright (C) 2011 Beat Röthlisberger
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/

function [fc, xc, output] = bfgs_min(FUN, GRAD, X0, varargin)

    output = struct;

    if (nargin == 4)
        
        opts = varargin{1};
        [ftol xtol gtol maxiter] = get_termination_criteria(opts);
    else
        
        [ftol xtol gtol maxiter] = get_termination_criteria();
    end
        
    STEPSIZE = 1;

    xc = X0;
    fc = FUN(xc);
    gc = GRAD(xc);
    pc = -gc;
    
    output.fvals(1) = fc;
    output.xvals{1} = xc;
    
    H = eye(length(xc));
    
    for iter = 1:maxiter
        
        % Store previous values
        xp = xc;
        fp = fc;
        gp = gc;     
        
        alpha = minimize1d_lin(FUN, GRAD, xc, pc, STEPSIZE);
        
        if (alpha == 0)
            
            warning('Line search could not make further progress.');
            output.status = -1;
            return;
        end
        
        % Update current values
        xc = xc + alpha*pc;

        if mod(iter, 20) == 0
        
            xc = mod(xc, 2*pi);
        end        
        
        fc = FUN(xc);
        gc = GRAD(xc);

        output.fvals(iter + 1) = fc;
        output.xvals{iter + 1} = xc;
        
        disp(['iter = ', num2str(iter), ' f = ', num2str(fc,15)]);
        
        dx = xc - xp;
        df = fc - fp;
        dg = gc - gp;

        if (norm(dg) < gtol)
            
            output.status = 0;
            return;
        end

        if (abs(df) < ftol)
            
            output.status = 1;
            return;
        end
        
        if (norm(dx) < xtol)
            
            output.status = 2;
            return;
        end
        
        dxdg = dx*dg';
        
        if (dxdg > 10*eps)

            hdg = dg*H;
            dghdg = dg*hdg';
            
            H = H + (1 + dghdg/dxdg)*(dx'*dx)/dxdg - (dx'*hdg + hdg'*dx)/dxdg;
        else
            
            H = eye(length(xc));
        end
        
        pc = -gc*H;
    end
    
    warning('Too many iterations.');
    output.status = -2;
end