%minimize1d_lin    One-parameter minimization along a line in Euclidean space.
%   a = minimize1d_lin(FUN, GRAD, X0, P, a0) performs a one-dimensional minimization
%   of FUN along the direction P passing through X0. More formally, it returns 
%   a local minimum of the function f(a) = FUN(X0 + a*P). GRAD must return the 
%   element-wise derivatives of FUN.
%   a0 >= 0 is an initial guess for the location of the minimum.
%   Important: The directional derivative of FUN at X0 (a = 0) must be
%   negative.

% Copyright (C) 2011 Beat Röthlisberger
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/


% The functions provided here are based on the C-File 'linear_minimize.c'
% from the GNU Scientific Library 'multimin' package (v1.14), written by Brian Gough.

function alpha = minimize1d_lin(FUN, GRAD, X0, P, alpha_start)
    
    alpha = alpha_start;

    RHO = 0.01;
    SIGMA = 0.1;
    TAU1 = 9;
    TAU2 = 0.05;
    TAU3 = 0.5;

    BRACKET_ITERS = 100;
    SECTION_ITERS = 100;

    f0 = FUN(X0);
    fp0 = GRAD(X0)*P';

    alpha_prev = 0;
    falpha_prev = f0;
    fpalpha_prev = fp0;

    a = 0;
    fa = f0;
    fpa = fp0;
    
    b = alpha;
    fb = 0;
    fpb = 0;


    for iter = 1:BRACKET_ITERS

        falpha = FUN(X0 + alpha*P);

        if (falpha > f0 + alpha*RHO*fp0 || falpha >= falpha_prev)

            a = alpha_prev; 
            fa = falpha_prev; 
            fpa = fpalpha_prev;

            b = alpha; 
            fb = falpha; 
            fpb = NaN;

            break;
        end

        fpalpha = GRAD(X0 + alpha*P)*P';

        if (abs(fpalpha) <= -SIGMA*fp0)

            return;
        end

        if (fpalpha >= 0)

            a = alpha; 
            fa = falpha; 
            fpa = fpalpha;

            b = alpha_prev; 
            fb = falpha_prev; 
            fpb = fpalpha_prev;

            break;
        end

        delta = alpha - alpha_prev;

        lower = alpha + delta;
        upper = alpha + TAU1*delta;

        alpha_next = interpolate(alpha_prev, falpha_prev, fpalpha_prev, alpha, falpha, fpalpha, lower, upper);

        alpha_prev = alpha;
        falpha_prev = falpha;
        fpalpha_prev = fpalpha;
        alpha = alpha_next;        
    end

    for iter = 1:SECTION_ITERS

        delta = b - a;
        
        lower = a + TAU2 * delta;
        upper = b - TAU3 * delta;
        
        alpha = interpolate(a, fa, fpa, b, fb, fpb, lower, upper);
        falpha = FUN(X0 + alpha*P);
      
        if ((a - alpha)*fpa <= eps) 

            % No further progress possible
            % Return 0.
            alpha = 0;
            return;
        end

        if (falpha > f0 + RHO * alpha * fp0 || falpha >= fa)
        
            b = alpha; 
            fb = falpha; 
            fpb = NaN;
        else
            
            fpalpha = GRAD(X0 + alpha*P)*P';
          
            if (abs(fpalpha) <= -SIGMA * fp0)
            
                return;
            end
          
            if (((b - a) >= 0 && fpalpha >= 0) || ((b - a) <=0 && fpalpha <= 0))
                
                b = a; 
                fb = fa; 
                fpb = fpa;
                
                a = alpha; 
                fa = falpha; 
                fpa = fpalpha;

            else
                a = alpha; 
                fa = falpha; 
                fpa = fpalpha;
            end
        end
    end
end





function alpha = interpolate(a, fa, fpa, b, fb, fpb, xmin, xmax)

    zmin = (xmin - a) / (b - a);
    zmax = (xmax - a) / (b - a);

    if (zmin > zmax)

        tmp = zmin;
        zmin = zmax;
        zmax = tmp;
    end
    
    if (isnan(fpb))
        
        z = interp_quad(fa, fpa*(b - a), fb, zmin, zmax);
    else

        z = interp_cubic(fa, fpa*(b - a), fb, fpb*(b - a), zmin, zmax);
    end
    
    alpha = a + z * (b - a);
end



function zmin = interp_quad(f0, fp0, f1, zl, zh)

    fl = f0 + zl*(fp0 + zl*(f1 - f0 -fp0));
    fh = f0 + zh*(fp0 + zh*(f1 - f0 -fp0));
    c = 2*(f1 - f0 - fp0);

    zmin = zl;
    fmin = fl;

    if (fh < fmin) 
        
        zmin = zh; 
        fmin = fh;
    end

    if (c > 0)
        
        z = -fp0/c;
        
        if (z > zl && z < zh) 

            f = f0 + z*(fp0 + z*(f1 - f0 -fp0));
        
            if (f < fmin)
                zmin = z; 
                fmin = f;
            end
        end
    end
end



function res = cubic(c0, c1, c2, c3, z)
    
    res = c0 + z*(c1 + z*(c2 + z*c3));
end



function [zmin, fmin] = check_extremum (c0, c1, c2, c3, z, zmin, fmin)
    
    y = cubic (c0, c1, c2, c3, z);

    if (y < fmin)
        
        zmin = z;
        fmin = y;
    end
end



function zmin = interp_cubic(f0, fp0, f1, fp1, zl, zh)

    eta = 3*(f1 - f0) - 2*fp0 - fp1;
    xi = fp0 + fp1 - 2*(f1 - f0);
    c0 = f0;
    c1 = fp0;
    c2 = eta;
    c3 = xi;

    zmin = zl; 
    fmin = cubic(c0, c1, c2, c3, zl);
    [zmin, fmin] = check_extremum(c0, c1, c2, c3, zh, zmin, fmin);

    tmp = c2^2 - 3*c1*c3;
    
    if (tmp < 0)
        return;
    else
        
        tmp = sqrt(tmp);
        z0 = (-c2 - tmp)/(3*c3);
        z1 = (-c2 + tmp)/(3*c3);
        
        if (z0 > zl && z0 < zh)
            
            [zmin, fmin] = check_extremum(c0, c1, c2, c3, z0, zmin, fmin);
        end
        
        if (z1 > zl && z1 < zh) 
          
            [zmin, fmin] = check_extremum(c0, c1, c2, c3, z1, zmin, fmin);
        end
    end
end