%cg_min   Conjugate-gradient minimization on the unitary group.
%   [f, U, info] = cg_min(FUN, GRAD, U0) performs a conjugate-gradient
%   minimization of the function FUN over the unitary group. GRAD must
%   return in its real and imaginary part the element-wise derivatives of 
%   FUN with respect to real(U) and imag(U), where U is a unitary matrix
%   and is FUN's and GRAD's only argument. The search is started at U0.
%
%   [f, U, info] = cg_min(FUN, GRAD, U0, params) is used to specify
%   termination parameters different from the default values. These
%   parameters in the struct params are (with default values in parentheses):
%
%   params.MaxIter (1000) Maximum number of iterations
%   params.TolFun (1e-12) Minimum difference between two consecutive
%                         function values
%   params.TolG (1e-10)   Minimum difference between two consecutive
%                         gradient norms
%   params.TolX (1e-10)   Minimum difference between two consecutive search 
%                         point norms
%
%   The return values f and U store the minimum function value and the
%   corresponding unitary matrix, respectively. The struct info contains:
%
%   info.fvals   The function values encountered during the minimization
%   info.xvals   A cell array containing the corresponding search points
%   info.status  Termination status, where
%
%                -2 = Too many iterations
%                -1 = Line search could not make further progress
%                 0 = Terminated on gradient tolerance
%                 1 = Terminated on function tolerance
%                 2 = Terminated on search point tolerance

% Copyright (C) 2011 Beat Röthlisberger
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/

function [fc, Tc, output] = cg_min(FUN, GRAD, T0, varargin)

    output = struct;

    if (nargin == 4)
        
        opts = varargin{1};
        [ftol xtol gtol maxiter] = get_termination_criteria(opts);
    else
        
        [ftol xtol gtol maxiter] = get_termination_criteria();
    end

    STEPSIZE = 1;
    
    Tc = T0;
    fc = FUN(Tc);
    gc  = aux_gradient(GRAD, Tc);

    output.fvals(1) = fc;
    output.xvals{1} = Tc;

    xc = -gc;

    for iter = 1:maxiter

        % Store previous values
        Tp = Tc;
        fp = fc;
        gp = gc;     
        
        alpha = minimize1d_exp(FUN, GRAD, Tc, xc, STEPSIZE);

        if (alpha == 0)

            warning('Line search could not make further progress.');
            output.status = -1;
            return;
        end
        
        % completeGramSchmidt increases numerical stability
        Tc = completeGramSchmidt(Tc*expm(alpha*xc));

        fc = FUN(Tc);
        gc  = aux_gradient(GRAD, Tc);

        output.fvals(iter + 1) = fc;
        output.xvals{iter + 1} = Tc;        
        
        dT = Tc - Tp;
        df = fc - fp;
        dg = gc - gp;

        if (norm(dg) < gtol)
            
            output.status = 0;
            return;
        end

        if (abs(df) < ftol)
            
            output.status = 1;
            return;
        end
        
        if (norm(dT) < xtol)
            
            output.status = 2;
            return;
        end        
        
        disp(['i = ' num2str(iter), ', f = ' num2str(fc, 15), '  |  alpha = ', num2str(alpha, 3)]);

        tau = expm(alpha/2*xc)*gp*expm(-alpha/2*xc);
        gamma = trace((gc - tau)*gc')/trace(gp*gp');
        xc = -gc + gamma*xc;

    end
    
    warning('Too many iterations.');
    output.status = -2;
end



function G = aux_gradient(GRAD, T)

    der = GRAD(T);
    der(:, (size(der, 2)+1):size(T, 1)) = 0;

    A = real(T.')*real(der) + imag(T.')*imag(der);
    B = real(T.')*imag(der) - imag(T.')*real(der);
    
    G = 1/2*(A - A' + 1i*(B + B'));
end
