%eofIsotropic    Entanglement of formation of an 'isotropic' density matrix.
%   RHO = eofIsotropic(f, d) calculates the entanglement of formation of an 
%   'isotropic' state of two qudits, each of dimension d, with mixing 
%   parameter 0 <= f <= 1.
%
%   See [B. M. Terhal and K. G. H. Vollbrecht, Phys. Rev. Lett. 85, 2625
%   (2000)].

% Copyright (C) 2011 Beat Röthlisberger
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/

function eof = eofIsotropic(f, d)

    if (f < 1/d)

        eof = 0;
    
    elseif (f < 4*(d - 1)/d^2)

        eof = r_function(f, d, 1, d - 1);

    else

        eof = d*log2(d - 1)/(d - 2)*(f - 1) + log2(d);
    end

end


function res = r_function(f, d, n, m)

    g_nm = gamma(f, d, n, m);

    res = h2(n*g_nm) + n*g_nm*log2(n/m) + log2(m);

    res = real(res);
end


function h = h2(x)
       
    if (x < eps || abs(x - 1) < eps)
        
        h = 0;
    else
        
        h = -x*log2(x) - (1-x)*log2(1-x);
    end
end
    

function g = gamma(f, d, n, m)

    g = ((sqrt(d*f)*n + sqrt(m*n*(m + n - d*f)))/(n*(n + m)))^2;
end
